#!/bin/sh

# Mount any special filesystems we need
mount_filesystems()
{
    echo "Mounting filesystems..."   
    
    # Mount the app/event on rwfs (ramdisk)  
    mkdir /mnt/rwfs/event
    mount -n -o bind /mnt/rwfs/event ${APPPATH}/event
}

# Work out what the default application is
get_default_app()
{
    # Check if the $CONFIGPATH/defaultapp exists  
    if [ -f ${CONFIGPATH}/defaultapp ]; then
        # It does, read it in
        DEFAULTAPP=$(cat ${CONFIGPATH}/defaultapp)
    fi
    
    # Make sure we have a default app    
    if [ -z ${DEFAULTAPP} ]; then
        # Its empty. Default to APP1
        DEFAULTAPP=${APP1}
    fi
    
    # Make sure that the default app exists. This also handles the case 
    # of the defaultapp file getting corrupted   
    if [ "${DEFAULTAPP}" != ":DEBUGMODE:" ] && [ ! -f ${BINPATH}/${DEFAULTAPP} ]; then
        # It doesn't exist. Default to APP1
        DEFAULTAPP=${APP1}
    fi
}

# Save the default app to $CONFIGPATH/defaultapp 
save_default_app()
{
    # Do we already have a 'defaultapp' file?
    if [ -f ${CONFIGPATH}/defaultapp ]; then
        # Yes. Is it set to the same as our current default app?
        if [ "$(cat ${CONFIGPATH}/defaultapp)" != "${DEFAULTAPP}" ]; then
            # No. Save the new defaultapp
            echo "${DEFAULTAPP}" > ${CONFIGPATH}/defaultapp
            sync
        fi
    else
        echo "${DEFAULTAPP}" > ${CONFIGPATH}/defaultapp
        sync
    fi
}

# Setup hotplug
setup_hotplug()
{
    cp ${BINPATH}/hotplug /media
    echo /media/hotplug > /proc/sys/kernel/hotplug
}

# Setup keypad GPIO's as inputs - Only used by 3.5"
setup_keypad()
{
    echo 68 > /sys/class/gpio/export
    echo in > /sys/class/gpio/gpio68/direction
    echo 69 > /sys/class/gpio/export
    echo in > /sys/class/gpio/gpio69/direction
    echo 71 > /sys/class/gpio/export
    echo in > /sys/class/gpio/gpio71/direction
    echo 99 > /sys/class/gpio/export
    echo in > /sys/class/gpio/gpio99/direction
    echo 70 > /sys/class/gpio/export
    echo in > /sys/class/gpio/gpio70/direction
}

# Touchscreen configuration
touchscreen_config()
{
	# Tslib parameters
	export TSLIB_TSDEVICE=/dev/input/event0
	export TSLIB_CONFFILE=${BINPATH}/ts.conf
	export TSLIB_CALIBFILE=${CONFIGPATH}/ts.calib
}

# Check for keys that are held during boot - Only used by 3.5"
check_for_held_keys()
{
    # Check for key 1 pressed to launch $APP1
    if [ `cat /sys/class/gpio/gpio68/value` -eq "0" ]; then
        DEFAULTAPP=${APP1}
        return
    fi
    
    # Check for key 2 pressed to launch $APP2
    if [ ! -z ${APP2} ]; then
        if [ `cat /sys/class/gpio/gpio69/value` -eq "0" ]; then
            DEFAULTAPP=${APP2}
            return
        fi
    fi
    
    # Check for key 3 pressed to launch $APP3
    if [ ! -z ${APP3} ]; then
        if [ `cat /sys/class/gpio/gpio71/value` -eq "0" ]; then
            DEFAULTAPP=${APP3}
            return
        fi
    fi
    
    # Check for key 4 pressed to launch $APP4
    if [ ! -z ${APP4} ]; then
        if [ `cat /sys/class/gpio/gpio99/value` -eq "0" ]; then
            DEFAULTAPP=${APP4}
            return
        fi
    fi
    
    # Check for key 5 pressed to launch $APP5
    if [ ! -z ${APP5} ]; then
        if [ `cat /sys/class/gpio/gpio70/value` -eq "0" ]; then
            DEFAULTAPP=${APP5}
            return
        fi
    fi
}

# Update the etc mirror in /var
setup_etc_overlay()
{
    # If the etc overlay exists, apply it
    if [ -d ${APPPATH}/etc ]; then
        cp -rp ${APPPATH}/etc /var
    fi
}

# Load any drviers we need
load_drivers()
{
    # USB Ethernet Drivers
    [ "${DRIVER_USB_NET_ASIX}" == "Y" ] && modprobe asix
    [ "${DRIVER_USB_NET_DM9601}" == "Y" ] && modprobe dm9601
    [ "${DRIVER_USB_NET_SMSC75XX}" == "Y" ] && modprobe smsc75xx
    [ "${DRIVER_USB_NET_SMSC95XX}" == "Y" ] && modprobe smsc95xx
    [ "${DRIVER_USB_NET_MCS7830}" == "Y" ] && modprobe mcs7830
    [ "${DRIVER_USB_NET_AX88179_178A}" == "Y" ] && modprobe ax88179_178a

    # USB Serial Drivers
    [ "${DRIVER_USB_SERIAL_FTDI_SIO}" == "Y" ] && modprobe ftdi_sio
    [ "${DRIVER_USB_SERIAL_PL2303}" == "Y" ] && modprobe pl2303
}

# Configure the networking (ethernet)
setup_networking()
{
    # Is ethernet enabled in settings.sh?
    if [ "${ETHERNET_ENABLED}" == "Y" ]; then
        # Yes. Setup ifplugd to monitor plug events and call our action scripts
        ifplugd -i eth0 -fI -r /etc/ifplugd.action
    fi
}

# Setup for 3.5" Display
setup_product_0()
{
    setup_keypad
    
    get_default_app
    
    check_for_held_keys
    
    save_default_app
    
	# QT Parameters
	export QWS_DISPLAY=LinuxFb:mmWidth=85:mmHeight=64:0
	export QWS_SIZE=320*240
	export QWS_KEYBOARD=SDKQKeypadDriver:gpios=68,69,71,99,70
	export QT_QWS_FONTDIR=${APPPATH}/fonts
}

# Setup for 7" Display
setup_product_256()
{
    touchscreen_config
    
    get_default_app
    
    save_default_app

	# QT Parameters
	export QWS_DISPLAY=LinuxFb:mmWidth=85:mmHeight=64:0
	export QWS_SIZE=800*480
	export QT_QWS_FONTDIR=${APPPATH}/fonts
}

################################################################################
# Start of main script
################################################################################

# Redirect output to debug serial
exec 1>/dev/ttyAMA0
exec 2>/dev/ttyAMA0

# Setup paths
SCRIPTPATH=$(cd ${0%/*} && echo ${PWD})
APPPATH="/app"
BINPATH="${APPPATH}/bin"
DATAPATH="${APPPATH}/data"
CONFIGPATH="${DATAPATH}/config"

# Get the Product Id from OTP (0 for 3.5", 256 for 7")
PRODUCTID=$(printf "%d" $(cat /sys/uid/otp/HW_OCOTP_CUST1))

# Read our settings
source ${SCRIPTPATH}/settings.sh

setup_etc_overlay

setup_hotplug

mount_filesystems

load_drivers

# Call the product specific setup function
setup_product_${PRODUCTID}

setup_networking

# Is DEBUGMODE requested?
if [ "${DEFAULTAPP}" != ":DEBUGMODE:" ]; then
    # No. 

    # Disable watchdog if requested    
    if [ "${WATCHDOG_OFF}" != "" ]; then
        export WATCHDOG_OFF
    fi        
    
    # Launch the app
	cd ${BINPATH}
    ./${DEFAULTAPP} -qws &
else
	# Debug mode. Dont launch the app.
	# Mount ${APPPATH} read-write
    mount -o remount,rw ${APPPATH}
    # and change the screen red
    ${BINPATH}/fbfill f800		
fi
